function spring_mass_running_grf
%% Tabula Rasa
clear all
close all


%% Parameters

l0 = 1;         % uncompressed spring length [m]
m = 80;         % mass [kg]
k = 23000;		% spring stiffness [N/m]
g = 9.81;		% acceleration due to gravity [m/s^2]
alpha0 = 70;	% landing angle of leg [deg]

alpha0 = deg2rad(alpha0);

%% Initial Conditions

x = 0;
y = 1;
x_dot = 5;
y_dot = 0;

x1 = -l0*cos(alpha0);  % horizontal distance between mass and foot contact at start of stance phase

%% State variable
q = [x y x_dot y_dot]'; % initial conditions

traj = [];

%% Transition Conditions

foot_impact = odeset('Events', @touch_down);    % event condition for landing
push_off = odeset('Events', @take_off);         % event condition for take off
peak = odeset('Events', @apex);                 % event contition for apex

%% Simulation
for i = 1:11
    [~,q] = ode45(@EoM_air, [0,5], q,foot_impact);  % integrate flight phase from apex
    traj = [traj; q];                               % store trajectory
    q = [x1 traj(end,2:4)]';                        % set initial conditions for stance phase, set origin to foot contact
    
    [t,q] = ode45(@EoM_ground, [0,5], q,push_off);  % integrate stance phase
%     Step(i).t = t;
%     Step(i).q = q;
    q(:,1) = q(:,1) + traj(end,1) + abs(x1);        % correct trajectory to include x displacement
    traj = [traj; q];                               % store trajectory
    q = traj(end,:);                                % set initial conditions for flight phase
  
    [~,q] = ode45(@EoM_air, [0,5], q,peak);         % integrate flight phase to apex
    traj = [traj; q];                               % store trajectory
    q = traj(end,:)';                               % set initial conditions for next step
end

%% Phase Plot
figure(31)
clf
plot(traj(:,1),traj(:,2))
title(['Trajectory of ' num2str(10) ' steps for spring mass running'])
xlabel('x (m)')
ylabel('y (m)')

%% Output Figure - uncomment to print a .eps file.
% print spring_mass_running_traj.eps -depsc2

% %% Ground reaction force
% % Calculate accelerations for last step:
% for t = 1:length(Step(end).t)
%     q_dot_temp = EoM_ground(1,Step(end).q(t,:));
%     pos_dot(t,1:2) = q_dot_temp(1:2)';
%     pos_ddot(t,1:2) = q_dot_temp(3:4)';
%     pos(t,1:2) = Step(end).q(t,1:2);
% end
% 
% % Ground reaction forces
% Fx = m*pos_ddot(:,1);
% Fy = m*pos_ddot(:,2) + m*g;
% 
% % Normalize with body weight
% Fx = Fx/(m*g);
% Fy = Fy/(m*g);
% 
% % Plot
% h_grf = figure(32);
% clf
% hold on
% t = Step(end).t/Step(end).t(end);
% plot(t,Fy,'k');
% plot(t,Fx,'b');
% xlabel('relative step time [-]')
% ylabel('relative ground reaction force [-]')
% legend('F_y','F_x')


%% Equation of Motion for Flight Phase
function output = EoM_air(~,q)
    output(1,1) = q(3);
    output(2,1) = q(4);
    output(3,1) = 0;
	output(4,1) = -g;
end
%% Equation of Motion for Stance Phase

% equations are in cartesian coordinates, makes switching between flight and
% stance phase easier, taken from 'The Spring-Mass Model for Running and
% Hopping' by R. Blickhan though you can derive it yourself easily enough

function output = EoM_ground(~,q)
    omega = sqrt(k/m);
	output(1,1) = q(3);
	output(2,1) = q(4);
	output(3,1) = q(1)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1);
	output(4,1) = q(2)*omega^2*(l0/sqrt(q(1)^2 + q(2)^2) - 1) - g;
end
%% Conditions for step completion
% event for apex, y_dot = 0
function [impact,terminate,direction] = apex(~,q)
    impact = q(4);
	terminate = 1;
	direction = -1;
end
%% Conditions for take off
% event for take-off, spring length equals natural length l0^2 = x^2 + y^2
function [impact,terminate,direction] = take_off(~,q)
    impact = sqrt(q(1)^2 + q(2)^2) - l0;
	terminate = 1;
	direction = 1;
end
%% Conditions for touch down
% Condition for touch down given touch down angle alpha
function [impact,terminate,direction] = touch_down(~,q)
	impact = q(2) - l0*sin(alpha0);
	terminate = 1;
	direction = -1;
end
end